/*
	HEJPEG.C

	Image file (JPEG format) display routines, including the main
	function called from the engine:

		hugo_displaypicture

	Plus the non-portable QuickC functions:

		hugo_initpalette
		hugo_initJPEGimage

		hugo_writescanline
		hugo_putpixel

	And adapted from the IJG library:

		read_JPEG_file

	for the Hugo Engine

	Copyright (c) 1995-2006 by Kent Tessman

	The routines in this file are based in part on the work of the
	Independent JPEG Group.  They are taken from release 6a (7-Feb-96).

	Basically, hugo_displaypicture(file) attempts to read and
	display the JPEG-compressed picture stored in <file>.  The file
	is passed hot, i.e., opened and positioned.  The image should fit
	the screen area described by the current text window.  (This
	calculation is performed by hugo_initJPEGimage().)

	This file assumes it will find a compiled IJG library and the
	necessary headers "jpeglib.h", "jconfig.h", and "jmorecfg.h".

	For what it's worth, if you're using not using some faster,
	OS-integrated JPEG-display mechanism, this should be easily
	portable to any operating system mainly by replacing
	hugo_initpalette(), hugo_putpixel() and possibly
	hugo_writescanline().
*/

#include "heheader.h"

int hugo_displaypicture(FILE *infile, long len);


#if defined (GRAPHICS_SUPPORTED) && !defined (IOTEST)

#include <setjmp.h>

#include <graph.h>      /* QuickC graphics fancies      */
#include <bios.h>       /* BIOS interrupt services      */

#include "jpeglib.h"


/* More function prototypes: */
int hugo_initpalette(void);
void hugo_initJPEGimage(j_decompress_ptr cinfo);
void hugo_writescanline(j_decompress_ptr cinfo, JSAMPLE *jbuffer);
void hugo_putpixel(int x, int y, int color);
int read_JPEG_file(FILE *infile);


int SCREENCOLORS;
int display_type = 0, color_space, mapped_color_space;

int HUGO_PALETTE[16];                   /* for Hugo text colors */

/* Display types */
extern const VGAHIGH_DT, VGALOW_DT, NONE_DT;


/* hugo_displaypicture

	(This is the only routine that gets called from the engine proper.)

	Assumes that "filename" is hot--i.e., opened and positioned--but
	closes it before returning.

	Loads and displays a JPEG picture, sizing it to the screen area
	described by the current text window.  hugo_displaypicture() could
	call any OS-specific loading mechanism to accomplish this; here,
	it depends on auxiliary functions such as hugo_initJPEGimage()
	to properly size and position the image.  If smaller than the
	current text window, the image should be centered, not stretched.

	Ultimately, graphic formats other than JPEG may be supported,
	hence the identification of the type flag and the switch
	at the end.

	Note that the len argument isn't used by this decompression
	routine, but it is passed in case another method might require it.
	
	Returns false if it fails because of an ERROR.
*/

int hugo_displaypicture(FILE *infile, long len)
{
	int type;

	/* (This display_type check is QuickC-specific) */
	if (!display_type || display_type==NONE_DT)
	{
		fclose(infile);
		return true;		/* not an error */
	}

	/* Loading/display can fail at any of the following: */

	if ((type = fgetc(infile))==EOF) goto Failed;

	if (fseek(infile, -1, SEEK_CUR)) goto Failed;

	switch (type)
	{
		case 0xFF:      /* JPEG */
			if (!read_JPEG_file(infile)) return 0;
			break;

		default:        /* unrecognized */
#if defined (DEBUGGER)
			SwitchtoDebugger();
			DebugMessageBox("Picture Loading Error", "Unrecognized graphic format");
			SwitchtoGame();
#endif
			goto Failed;
	}

	return 1;

	/* read_JPEG_file() closes the file regardless of success or
	   failure; otherwise, it must be closed here
	*/
Failed:
	fclose(infile);
	return 0;
}


/* hugo_initpalette

	Does necessary initializations for number of colors and whether
	or not we're forcing to grayscale.  Basically, we have to
	generate a palette of n possible colors, where n is 16 or 256
	in VGA modes--although the actual number of colors recognized by
	the library will be 8 or 216, respectively.
*/

/* Convert RGB to a QuickC long color: */
#define RGB(red, green, blue) (((long)((blue)<<8|(green))<<8)|(red))

int hugo_initpalette(void)
{
	int i, inc, original_screencolors;
	int rgb_brightness = 0;
	long pal[256], red, green, blue;
	struct videoconfig vc;

	/* Re-establish the basic Hugo color set */
	for (i=0; i<16; i++)
	{
		HUGO_PALETTE[i] = i;
	}

	if (!display_type || display_type==NONE_DT) return 0;

	_getvideoconfig(&vc);

	original_screencolors = SCREENCOLORS = vc.numcolors;

	/* First, make sure all palette numbers are valid */
	for (i=0; i<SCREENCOLORS; i++)
		pal[i] = 0;

	/* Because the IJG library can't quantize to less than 16 colors
	   (or actually 8, but 16 is compatible across modes)
	*/
	if (SCREENCOLORS < 16) SCREENCOLORS = 16;

	/* Color... */
	if (color_space==JCS_RGB)
	{
	  switch (SCREENCOLORS)    /* Active bits in this order: */
	  {
	     case 256:
		inc = 12;          /* ???????? ??bbbbbb ??gggggg ??rrrrrr */
		break;

	     case 16:
	     default:
		inc = 32;          /* ???????? ??Bb???? ??Gg???? ??Rr???? */
		rgb_brightness = 12;
		break;
	  }

	  i = 0;
	  for (red   = 0;   red < 64;   red += inc)
	  for (green = 0; green < 64; green += inc)
	  for (blue  = 0;  blue < 64;  blue += inc)
	  {
		/* Make sure black stays black, not gray */
		if (i<SCREENCOLORS/16)
			pal[i] = RGB(red, green, blue);
		else
			pal[i] = RGB(red   + rgb_brightness,
				     green + rgb_brightness,
				     blue  + rgb_brightness);

		/* Special case where a total of 6 bits represent 16
		   colors:  if the first bit of each three sets is lit,
		   the color is on; if the second is lit, the color is
		   bright
		*/
		if (inc==32) pal[i+8] = pal[i] | pal[i]>>1;

		i++;
	  }

	  /* SCREENCOLORS will have changed to 8 or 216, both of which
	     have cube-roots and can therefore represent an RGB array
	  */
	  SCREENCOLORS = i;

	}

	/* ...or grayscale */
	else
	{
		for (i=0; i<SCREENCOLORS; i++)
		{
			/* Have to do some conversions, since RGB palette
			   elements range from 0-64
			*/
			if (SCREENCOLORS==16)
				red = i*4;
			else if (SCREENCOLORS==256)
				red = i/4;

			pal[i] = RGB(red, red, red);
		}
	}

	/* If we've changed from the original Hugo palette mapping,
	   remap HUGO_PALETTE[0..16]
	*/
	if (SCREENCOLORS>=216)
	{
		if (color_space==JCS_RGB)
		{
			i = 216;
			inc = 32;
			rgb_brightness = 0;	/* for pal[216] */
			for (red   = 0;   red < 64;   red += inc)
			for (green = 0; green < 64; green += inc)
			for (blue  = 0;  blue < 64;  blue += inc)
			{
				pal[i] = RGB(red   + rgb_brightness,
					     green + rgb_brightness,
					     blue  + rgb_brightness);

				if (inc==32) pal[i+8] = pal[i] | pal[i]>>1;
				i++;

				rgb_brightness = 12;
			}
			for (i=0; i<16; i++) HUGO_PALETTE[i] = 216+i;
		}

		else    /* grayscale */
		{
			for (i=0; i<16; i++) HUGO_PALETTE[i] = i*16;
		}
	}

	if (color_space==JCS_RGB)
	{
		/* Make brown "brown" instead of "mustard"... */
		pal[6 + (SCREENCOLORS==216?216:0)] = RGB(36, 28, 20);

		/* ...and make white a little whiter */
		pal[7 + (SCREENCOLORS==216?216:0)] = RGB(46, 46, 46);
	}

	/* No good reason why this should fail, but just in case */
	if (!_remapallpalette(pal)) return 0;

	hugo_settextcolor(DEF_FCOLOR);
	hugo_setbackcolor(DEF_BGCOLOR);

	mapped_color_space = color_space;

	return 1;
}


/* hugo_initJPEGimage

	Fits the JPEG image to the current text window, either scaling
	it down or centering it as necessary.
*/

float cscaling, rscaling;       /* column and row scaling */
int col_offset;                 /* column offset, if any  */
int row;                        /* last-drawn row         */
int row_stride;                 /* number bytes/row       */

void hugo_initJPEGimage(j_decompress_ptr cinfo)
{
	float maxheight, maxwidth;              /* of screen */
	float height, width;                    /* of image */

	/* Default is no horizontal or vertical scaling */
	cscaling = rscaling = 1;
	row = 0;

	/* First of all, figure what the maximum allowable size of the image
	   is (i.e., the current text window), and coerce the drawing width
	   and height to fit it
	*/
	maxheight = physical_windowheight;
	maxwidth = physical_windowwidth;

	height = (int)(cinfo->output_height);
	if (height>maxheight)
	{
		rscaling = maxheight/height;
		height = maxheight;
	}

	width = (int)(cinfo->output_width);
	if (width>maxwidth)
	{
		cscaling = maxwidth/width;
		width = maxwidth;
	}

	if (cscaling > rscaling) cscaling = rscaling;
	if (rscaling > cscaling) rscaling = cscaling;

	/* Convert 1.60:1 to 1.33:1 aspect ratio if necessary */
/*
	if (display_type==VGALOW_DT) rscaling*=.83;
*/

	/* Now, figure out both a screen row and a column offset--if
	   the image is smaller than the currently defined window,
	   we want to center it
	*/
	row = physical_windowtop + 
		(int)(maxheight/2) - (int)(cinfo->output_height*rscaling/2);

	row/=rscaling;
	row++;

	col_offset = physical_windowleft +
		(int)(maxwidth/2) - (int)(cinfo->output_width*cscaling/2);

	/* cinfo->output_components is 3 if RGB, 1 if quantizing to a
	   color map or grayscale--ergo, in this implementation, it is
	   always 1:
	*/
	row_stride = cinfo->output_width * cinfo->output_components;
}


/* hugo_writescanline */

void hugo_writescanline(j_decompress_ptr cinfo, JSAMPLE *jbuffer)
{
	int j, x = 0;
	int color;
	static int last_row;

	/* Skim through the scanline, getting color indexes and firing
	   them out pixel-by-pixel--but only bother if we're not
	   simply drawing over a previous row (i.e., when the image is
	   larger than the display area)
	*/
	if ((int)(row*rscaling) != (int)(last_row*rscaling))
	{
		/* Since we're quantizing to a color index (instead of
		   three RGB components), row_stride is equal to the
		   image width, and output_components is 1
		*/
		for (j=0; j<row_stride; j+=cinfo->output_components)
		{
			color = *(jbuffer + j);
			hugo_putpixel(col_offset+(int)(x++*cscaling), (int)(row*rscaling), color);
		}
	}

	last_row = row++;
}


/* hugo_putpixel

	Uses a BIOS interrupt in order to speed things up a little.
	Depends on having gotten good parameters, since no clipping/
	adjustment is done.
*/

void hugo_putpixel(int x, int y, int color)
{
	unsigned char asbyte, cbyte;

	/* So that in 16-color VGAHIGH mode (8 colors in the color cube),
	   we use the bright colors, not the dim ones
	*/
	if ((color_space==JCS_RGB) && display_type==VGAHIGH_DT && color)
		color += 8;

#if defined (DEBUGGER)
	asbyte = (unsigned char)active_screen;
#endif
	cbyte = (unsigned char)color;

	_asm
	{
		mov ah, 0Ch             /* function */
#if defined (DEBUGGER)
		mov bh, asbyte		/* video page */
#else
		mov bh, 0h
#endif
		mov al, cbyte
		mov cx, x
		mov dx, y

		int 10h                 /* plot the pixel */
	}
}


/*---------------------------------------------------------------------------
   JPEG Decompression Interface (from the IJG library):

   For further elaboration on the how and why of read_JPEG_file(), see
   "libjpeg.doc" with the IJG distribution.  For the sake of brevity
   and relevance to the Hugo implementation, some of the commenting has
   been trimmed or modified.
----------------------------------------------------------------------------*/

/* First of all, since a decompression error (memory or who knows what)
   may occur deep in the bowels of the IJG library, we need a way to
   harmlessly recover.  Hugo's default behavior is simply to abort loading
   the JPEG without reporting an error.

   The IJG library allows the user to build a hook into its normal
   error handler using setjmp() and longjmp(), which will instantly
   switch control to the blissfully ignorant return in read_JPEG_file(),
   below.
*/

struct hejpeg_error_mgr
{
	struct jpeg_error_mgr pub;      /* "public" fields */
	jmp_buf setjmp_buffer;          /* for return to caller */
};

typedef struct hejpeg_error_mgr *hejpeg_error_ptr;

void hejpeg_error_exit(j_common_ptr cinfo)
{
	/* cinfo->err really points to a hejpeg_error_mgr struct, so
	   coerce pointer
	*/
	hejpeg_error_ptr myerr = (hejpeg_error_ptr) cinfo->err;

	/* This is where the IJG library displays the error message--Hugo
	   doesn't, since it returns (even in mid-failed-decompression) to
	   the calling function

		(*cinfo->err->output_message) (cinfo);
	*/

	/* Return control to the setjmp point */
	longjmp(myerr->setjmp_buffer, 1);
}


/* read_JPEG_file

	Assumes the file passed as infile has been opened successfully.
*/

int read_JPEG_file(FILE *infile)
{
	/* This struct contains the JPEG decompression parameters and
	 * pointers to working space (which is allocated as needed by
	 * the JPEG library).
	 */
	struct jpeg_decompress_struct cinfo;

	/* We use our private extension JPEG error handler.
	 * Note that this struct must live as long as the main JPEG parameter
	 * struct, to avoid dangling-pointer problems.
	 */
	struct hejpeg_error_mgr jerr;

	JSAMPARRAY jbuffer;     /* output row buffer */
	int row_stride;         /* physical row width in output buffer */


/* Step 1: Allocate and initialize JPEG decompression object */

	/* We set up the normal JPEG error routines, then override
	   error_exit.
	*/
	cinfo.err = jpeg_std_error(&jerr.pub);
	jerr.pub.error_exit = hejpeg_error_exit;

	/* Establish the setjmp return context for hejpeg_error_exit
	   to use.
	*/
	if (setjmp(jerr.setjmp_buffer))
	{
	    /* If we get here, the JPEG code has signaled an error.
	     * We need to clean up the JPEG object, close the input file,
	     * and return.
	     */
	    jpeg_destroy_decompress(&cinfo);
	    fclose(infile);
	    return 0;
	}

	/* Now we can initialize the JPEG decompression object. */
	jpeg_create_decompress(&cinfo);

/* Step 2: Specify the data source */

	/* Return values from most library initializations are ignored
	   since suspension is not possible from the stdio data source
	*/
	jpeg_stdio_src(&cinfo, infile);

/* Step 3: Read file parameters with jpeg_read_header() */

	jpeg_read_header(&cinfo, TRUE);

/* Step 4: set parameters for decompression */

	/* Quantize down from 24-bit color to whatever we can muster */
	cinfo.quantize_colors = TRUE;
	if (cinfo.num_components > 1)
		cinfo.out_color_space = mapped_color_space;

	/* Let the library build a suitable color map instead of supplying
	   it with one
	*/
	cinfo.colormap = NULL;
	cinfo.two_pass_quantize = FALSE;

	/* Choose Floyd-Steinberg dithering and give the library a color
	   count to quantize to
	*/
	if (display_type==VGAHIGH_DT && color_space==JCS_RGB)
		cinfo.dither_mode = JDITHER_NONE;
	else
		cinfo.dither_mode = JDITHER_FS;
	cinfo.desired_number_of_colors = SCREENCOLORS;

	/* JDCT_DEFAULT (JDCT_ISLOW) is slower/higher quality; JDCT_FASTEST
	   is JDCT_IFAST by default
	*/
	cinfo.dct_method = JDCT_ISLOW; /* was JDCT_FASTEST */

	/* Little quality cost to forgo fancy upsampling */
	cinfo.do_fancy_upsampling = FALSE;

/* Step 5: Start decompressor */

	jpeg_start_decompress(&cinfo);

	hugo_initJPEGimage(&cinfo);

	/* JSAMPLEs per row in output buffer */
	row_stride = cinfo.output_width * cinfo.output_components;

	/* Make a one-row-high sample array that will go away when
	   done with the image
	*/
	jbuffer = (*cinfo.mem->alloc_sarray)
		((j_common_ptr)&cinfo, JPOOL_IMAGE, row_stride, 1);

/* Step 6: while (scan lines remain to be read) */

	/* Here we use the library's state variable cinfo.output_scanline
	 * as the loop counter, so that we don't have to keep track
	 * ourselves.
	 */
	while (cinfo.output_scanline < cinfo.output_height)
	{
		/* jpeg_read_scanlines expects an array of pointers
		 * to scanlines.  Here the array is only one element
		 * long, but you could ask for more than one scanline
		 * at a time if that's more convenient.
		 */
		jpeg_read_scanlines(&cinfo, jbuffer, 1);

		hugo_writescanline(&cinfo, jbuffer[0]);

	}

/* Step 7: Finish decompression */

	jpeg_finish_decompress(&cinfo);

/* Step 8: Release JPEG decompression object */

	jpeg_destroy_decompress(&cinfo);

	fclose(infile);

	return 1;
}

#else   /* if !defined (GRAPHICS_SUPPORTED) || defined (IOTEST) */

int hugo_displaypicture(FILE *infile, long len)
{
	fclose(infile);         /* since infile will be open */

	return 1;
}
#endif

/* For convenience in building iotest.exe with QuickC: */
#if defined (IOTEST)
#include <graph.h>
/* From hejpeg.h: */
int display_type = 0;
/* Constants from graph.h: */
const VGAHIGH_DT = _HRES16COLOR;
const VGALOW_DT = _MRES256COLOR;
const NONE_DT = _TEXTC80;
#endif

